-- modules/game_item_tier_shaders/tier_shaders.lua
-- Applies a unique shader per item Tier (supports up to MAX_TIER).
-- Tier discovery order:
--   1) item:getTier() if present (best)
--   2) parse tier from tooltip text (fallback) e.g. "Tier: 3" / "Item Tier: 3" / "Tier 3"
-- If an item has tier <= 0 or unknown, shader is cleared.

TierItemShaders = TierItemShaders or {}

local MAX_TIER = 15

local function clampTier(t)
  t = tonumber(t) or 0
  if t < 0 then t = 0 end
  if t > MAX_TIER then t = MAX_TIER end
  return t
end

local function getTierFromItem(item)
  if not item then return 0 end

  -- 1) Direct API
  if item.getTier then
    local t = item:getTier()
    if t then return clampTier(t) end
  end

  -- 2) Tooltip parse fallback
  if item.getTooltip then
    local tip = item:getTooltip()
    if tip and #tip > 0 then
      local t = tip:match("[Ii]tem%s+[Tt]ier%s*:%s*(%d+)")
             or tip:match("[Tt]ier%s*:%s*(%d+)")
             or tip:match("[Tt]ier%s+(%d+)")
      if t then return clampTier(t) end
    end
  end

  return 0
end

local function shaderNameForTier(tier)
  tier = clampTier(tier)
  if tier <= 0 then return "" end
  return string.format("Item Tier %02d", tier)
end

local function registerTierShaders()
  for tier = 1, MAX_TIER do
    local name = string.format("Item Tier %02d", tier)
    local frag = string.format("/modules/game_item_tier_shaders/shaders/fragment/tier_%02d.frag", tier)

    -- Keep non-framebuffer for stability/perf in all UI contexts.
    g_shaders.createFragmentShader(name, frag, false)

    -- Items bind shaders by name; routing through an existing setup pipeline is sufficient.
    if g_shaders.setupOutfitShader then
      g_shaders.setupOutfitShader(name)
    end
  end
end

local function applyTierShaderToWidget(widget)
  if not widget or widget:isDestroyed() then return end
  if not widget.getItem then return end

  local item = widget:getItem()
  if not item then return end

  local tier = getTierFromItem(item)
  local shader = shaderNameForTier(tier)

  if item.setShader then
    item:setShader(shader)
  end
end

local function hookUIItem()
  if not UIItem then return end

  if not TierItemShaders._oldSetItem then
    TierItemShaders._oldSetItem = UIItem.setItem
    UIItem.setItem = function(self, item)
      TierItemShaders._oldSetItem(self, item)
      applyTierShaderToWidget(self)
    end
  end

  if UIItem.setItemId and not TierItemShaders._oldSetItemId then
    TierItemShaders._oldSetItemId = UIItem.setItemId
    UIItem.setItemId = function(self, itemId)
      TierItemShaders._oldSetItemId(self, itemId)
      applyTierShaderToWidget(self)
    end
  end
end

function init()
  registerTierShaders()
  hookUIItem()
end

function terminate()
  if UIItem and TierItemShaders._oldSetItem then
    UIItem.setItem = TierItemShaders._oldSetItem
    TierItemShaders._oldSetItem = nil
  end
  if UIItem and TierItemShaders._oldSetItemId then
    UIItem.setItemId = TierItemShaders._oldSetItemId
    TierItemShaders._oldSetItemId = nil
  end
end
