g_logger.info("[game_weather] FILE LOADED")

local WEATHER_OPCODE = 208

local WEATHER_DEFAULT_SHADER = "Default"
local WEATHER_SHADERS = {
    rain = "Map - Rain",
    fog = "Map - Fog",
    snow = "Map - Snow",
    storm = "Map - Rain",
    none = WEATHER_DEFAULT_SHADER
}

WeatherController = Controller:new()

-- Simple rectangular zones. Replace with real coordinates.
local zones = {
    { name = "Thais", from = { x = 32369, y = 32241, z = 7 }, to = { x = 32399, y = 32270, z = 7 }, weather = "rain" },
    { name = "Venore", from = { x = 32950, y = 32060, z = 7 }, to = { x = 32980, y = 32090, z = 7 }, weather = "fog" }
}
local snowGroundIds = {}
local snowRegions = {}
do
    local ok, data = pcall(dofile, "zones")
    if ok and type(data) == "table" then
        if data.zones then
            zones = data.zones
            snowGroundIds = {}
            if type(data.snowGroundIds) == "table" then
                for _, id in ipairs(data.snowGroundIds) do
                    snowGroundIds[id] = true
                end
            end
            snowRegions = data.snowRegions or {}
        else
            zones = data
        end
    end
end

local state = {
    globalActive = false,
    globalType = nil,
    globalIntensity = 1.0,
    globalUntil = 0,
    globalForce = false,
    zoneType = nil,
    enabled = true,
    applied = nil,
    serverDisabled = false
}

local debounceEvent = nil
local globalTimerEvent = nil
local playerEventController = nil
local warnedMissingShaderTarget = false
local warnedMapNil = false

local function getMapPanel()
    if modules.game_interface and modules.game_interface.getMapPanel then
        return modules.game_interface.getMapPanel()
    end
    return nil
end

local function resolveShader(weatherType)
    local shaderName = WEATHER_SHADERS[weatherType] or WEATHER_DEFAULT_SHADER
    if shaderName ~= WEATHER_DEFAULT_SHADER and not g_shaders.getShader(shaderName) then
        g_logger.warning("[game_weather] Missing shader: " .. shaderName .. ", falling back to Default")
        shaderName = WEATHER_DEFAULT_SHADER
    end
    return shaderName
end

local function applyShader(shaderName)
    if state.applied == shaderName then
        return
    end

    local map = getMapPanel()
    if not map then
        if not warnedMapNil then
            g_logger.warning("[game_weather] map panel is nil; cannot apply shader.")
            warnedMapNil = true
        end
        return
    end
    if not map.setShader then
        if not warnedMissingShaderTarget then
            g_logger.warning("[game_weather] Map panel does not support setShader; weather effects disabled. map=" .. tostring(map))
            warnedMissingShaderTarget = true
        end
        return
    end

    g_logger.info("[game_weather] applying shader: " .. shaderName)
    map:setShader(shaderName)
    state.applied = shaderName
end

local function positionInZone(pos, zone)
    local minX = math.min(zone.from.x, zone.to.x)
    local maxX = math.max(zone.from.x, zone.to.x)
    local minY = math.min(zone.from.y, zone.to.y)
    local maxY = math.max(zone.from.y, zone.to.y)
    local minZ = math.min(zone.from.z, zone.to.z)
    local maxZ = math.max(zone.from.z, zone.to.z)

    return pos.x >= minX and pos.x <= maxX
        and pos.y >= minY and pos.y <= maxY
        and pos.z >= minZ and pos.z <= maxZ
end

local function updateZoneWeather()
    local player = g_game.getLocalPlayer()
    if not player then
        return
    end

    local pos = player:getPosition()
    if not pos then
        return
    end

    local zoneWeather = nil
    for _, zone in ipairs(zones) do
        if positionInZone(pos, zone) then
            zoneWeather = zone.weather
            break
        end
    end

    state.zoneType = zoneWeather
end

local function isSnowEligibleAtPlayerPosition()
    local player = g_game.getLocalPlayer()
    if not player then
        return false
    end

    local pos = player:getPosition()
    if not pos then
        return false
    end

    for _, zone in ipairs(snowRegions) do
        if positionInZone(pos, zone) then
            return true
        end
    end

    local tile = g_map and g_map.getTile and g_map.getTile(pos) or nil
    if not tile then
        return false
    end

    local ground = tile:getGround()
    if not ground then
        return false
    end

    local groundId = ground:getId()
    if not groundId then
        return false
    end

    return snowGroundIds[groundId] == true
end

local function clearGlobalWeather()
    state.globalActive = false
    state.globalType = nil
    state.globalIntensity = 1.0
    state.globalUntil = 0
    state.globalForce = false
end

local function scheduleGlobalClear(durationSeconds)
    if globalTimerEvent then
        removeEvent(globalTimerEvent)
        globalTimerEvent = nil
    end

    if durationSeconds and durationSeconds > 0 then
        globalTimerEvent = scheduleEvent(function()
            clearGlobalWeather()
            scheduleEvent(function()
                if WeatherController and WeatherController.applyWeather then
                    WeatherController:applyWeather()
                end
            end, 1)
        end, durationSeconds * 1000)
    end
end

function WeatherController:applyWeather()
    if not state.enabled then
        applyShader(WEATHER_DEFAULT_SHADER)
        return
    end

    local desired
    if state.serverDisabled then
        desired = "none"
    else
        desired = state.globalActive and state.globalType or state.zoneType or "none"
    end
    if desired == "snow" and not state.globalForce and not isSnowEligibleAtPlayerPosition() then
        desired = "none"
    end
    local shaderName = resolveShader(desired)
    applyShader(shaderName)
end

local function onPositionChange()
    if not state.enabled then
        return
    end
    if debounceEvent then
        return
    end

    debounceEvent = scheduleEvent(function()
        debounceEvent = nil
        updateZoneWeather()
        WeatherController:applyWeather()
    end, 250)
end

local function onExtendedWeather(protocol, opcode, data)
    if type(data) ~= "table" then
        return
    end

    if data.op ~= "weather" or data.scope ~= "global" then
        return
    end

    g_logger.info("[game_weather] JSON weather received: " .. json.encode(data))

    if data.active then
        state.serverDisabled = false
        state.globalActive = true
        state.globalType = data.type or "rain"
        state.globalIntensity = tonumber(data.intensity) or 1.0
        state.globalForce = data.force == true
        local duration = tonumber(data.duration) or 0
        if duration > 0 then
            state.globalUntil = g_clock.millis() + (duration * 1000)
            scheduleGlobalClear(duration)
        else
            state.globalUntil = 0
            scheduleGlobalClear(0)
        end
    else
        state.serverDisabled = true
        clearGlobalWeather()
        scheduleGlobalClear(0)
    end

    WeatherController:applyWeather()
end

local function readEnabledOption()
    if modules.client_options and modules.client_options.getOption then
        return modules.client_options.getOption("enableWeatherEffects")
    end
    if g_settings then
        return g_settings.getBoolean("enableWeatherEffects", true)
    end
    return true
end

function setEnabled(value)
    state.enabled = value == true
    if not state.enabled then
        applyShader(WEATHER_DEFAULT_SHADER)
        return
    end
    updateZoneWeather()
    WeatherController:applyWeather()
end

function WeatherController:onInit()
    g_logger.info("[game_weather] init, registering opcode=" .. WEATHER_OPCODE)
    ProtocolGame.registerExtendedJSONOpcode(WEATHER_OPCODE, onExtendedWeather)
    ProtocolGame.registerExtendedOpcode(WEATHER_OPCODE, function(protocol, opcode, buffer)
        g_logger.info("[game_weather] RAW opcode received len=" .. tostring(#buffer) .. " data=" .. tostring(buffer))
    end)
    state.enabled = readEnabledOption()
end

function WeatherController:onTerminate()
    g_logger.info("[game_weather] terminate cleanup")
    if playerEventController then
        playerEventController:destroy()
        playerEventController = nil
    end
    if debounceEvent then
        removeEvent(debounceEvent)
        debounceEvent = nil
    end
    if globalTimerEvent then
        removeEvent(globalTimerEvent)
        globalTimerEvent = nil
    end
    ProtocolGame.unregisterExtendedJSONOpcode(WEATHER_OPCODE)
    ProtocolGame.unregisterExtendedOpcode(WEATHER_OPCODE)
end

function WeatherController:onGameStart()
    g_logger.info("[game_weather] game start, enabling position hook")
    playerEventController = self:registerEvents(LocalPlayer, {
        onPositionChange = onPositionChange
    })
    playerEventController:execute()

    updateZoneWeather()
    self:applyWeather()
end

function WeatherController:onGameEnd()
    g_logger.info("[game_weather] game end cleanup")
    clearGlobalWeather()
    state.zoneType = nil
    applyShader(WEATHER_DEFAULT_SHADER)
    if playerEventController then
        playerEventController:destroy()
        playerEventController = nil
    end
end
