ItemsDatabase = {}

ItemsDatabase.rarityColors = {
    ["yellow"] = TextColors.yellow,
    ["purple"] = TextColors.purple,
    ["blue"] = TextColors.blue,
    ["green"] = TextColors.green,
    ["grey"] = TextColors.grey,
}

ItemsDatabase.lootColorConfig = ItemsDatabase.lootColorConfig or {
    pricesFile = "/prices.json",
    prefer = "max", -- "max" (default), "price", "tier"
    useMeanPriceFallback = true,
    tierThresholds = {
        { minTier = 10, rarity = "yellow" },
        { minTier = 7, rarity = "purple" },
        { minTier = 4, rarity = "blue" },
        { minTier = 2, rarity = "green" },
        { minTier = 1, rarity = "grey" },
    }
}

ItemsDatabase.priceOverrides = ItemsDatabase.priceOverrides or nil
ItemsDatabase._pricesLoaded = ItemsDatabase._pricesLoaded or false

local function getRarityRank(rarity)
    if rarity == "yellow" then return 5 end
    if rarity == "purple" then return 4 end
    if rarity == "blue" then return 3 end
    if rarity == "green" then return 2 end
    if rarity == "grey" then return 1 end
    return 0
end

local function getColorForValue(value)
    if value >= 1000000 then
        return "yellow"
    elseif value >= 100000 then
        return "purple"
    elseif value >= 10000 then
        return "blue"
    elseif value >= 1000 then
        return "green"
    elseif value >= 50 then
        return "grey"
    else
        return "white"
    end
end

local function clipfunction(value)
    if value >= 1000000 then
        return "128 0 32 32"
    elseif value >= 100000 then
        return "96 0 32 32"
    elseif value >= 10000 then
        return "64 0 32 32"
    elseif value >= 1000 then
        return "32 0 32 32"
    elseif value >= 50 then
        return "0 0 32 32"
    end
    return ""
end

local function getRarityForTier(tier)
    for _, entry in ipairs(ItemsDatabase.lootColorConfig.tierThresholds) do
        if tier >= entry.minTier then
            return entry.rarity
        end
    end
    return "white"
end

local function getPriceOverride(itemId)
    if not ItemsDatabase.priceOverrides then
        return nil
    end
    return ItemsDatabase.priceOverrides[itemId] or ItemsDatabase.priceOverrides[tostring(itemId)]
end

function ItemsDatabase.loadPriceOverrides()
    if ItemsDatabase._pricesLoaded then
        return
    end
    ItemsDatabase._pricesLoaded = true

    local file = ItemsDatabase.lootColorConfig.pricesFile
    if not file or file == "" or not g_resources.fileExists(file) then
        return
    end

    local ok, contents = pcall(g_resources.readFileContents, file)
    if not ok or not contents or contents == "" then
        return
    end

    local status, decoded = pcall(json.decode, contents)
    if not status or type(decoded) ~= "table" then
        return
    end

    local items = decoded.items or decoded
    if type(items) == "table" then
        ItemsDatabase.priceOverrides = items
    end

    if type(decoded.config) == "table" then
        if type(decoded.config.prefer) == "string" then
            ItemsDatabase.lootColorConfig.prefer = decoded.config.prefer
        end
        if type(decoded.config.useMeanPriceFallback) == "boolean" then
            ItemsDatabase.lootColorConfig.useMeanPriceFallback = decoded.config.useMeanPriceFallback
        end
    end
end

function ItemsDatabase.setRarityItem(widget, item, style)
    if not g_game.getFeature(GameColorizedLootValue) or not widget then
        return
    end
    local frameOption = modules.client_options.getOption('framesRarity')
    if frameOption == "none" then
        return
    end
    local imagePath = '/images/ui/item'
    local clip = nil
    if item then
        local price = type(item) == "number" and item or (item and item:getMeanPrice()) or 0
        local itemRarity = getColorForValue(price)
        if itemRarity then
            clip = clipfunction(price)
            if clip ~= "" then
                if frameOption == "frames" then
                    imagePath = "/images/ui/rarity_frames"
                elseif frameOption == "corners" then
                    imagePath = "/images/ui/containerslot-coloredges"
                end
            else
                clip = nil
            end
        end
    end
    widget:setImageClip(clip)
    widget:setImageSource(imagePath)
    if style then
        widget:setStyle(style)
    end
end

function ItemsDatabase.getColorForRarity(rarity)
    return ItemsDatabase.rarityColors[rarity] or TextColors.white
end

function ItemsDatabase.setColorLootMessage(text)
    ItemsDatabase.loadPriceOverrides()

    local function coloringLootName(match)
        local idString, rest = match:match("^(%d+)|(.+)$")
        if not idString then
            return "{" .. match .. "}"
        end

        local itemId = tonumber(idString)
        if not itemId then
            return "{" .. match .. "}"
        end

        local itemName, tierString = rest:match("^(.+)|(%d+)$")
        if not itemName then
            itemName = rest
        end

        local tier = tonumber(tierString) or 0

        local overridePrice = getPriceOverride(itemId)
        local meanPrice = nil
        if ItemsDatabase.lootColorConfig.useMeanPriceFallback then
            local thingType = g_things.getThingType(itemId, ThingCategoryItem)
            if thingType then
                meanPrice = thingType:getMeanPrice()
            end
        end

        local price = overridePrice or meanPrice

        local rarityFromPrice = price and getColorForValue(price) or "white"
        local rarityFromTier = tier > 0 and getRarityForTier(tier) or "white"

        local chosenRarity = "white"
        if ItemsDatabase.lootColorConfig.prefer == "price" then
            chosenRarity = rarityFromPrice ~= "white" and rarityFromPrice or rarityFromTier
        elseif ItemsDatabase.lootColorConfig.prefer == "tier" then
            chosenRarity = rarityFromTier ~= "white" and rarityFromTier or rarityFromPrice
        else
            chosenRarity = getRarityRank(rarityFromTier) > getRarityRank(rarityFromPrice) and rarityFromTier or rarityFromPrice
        end

        local color = ItemsDatabase.getColorForRarity(chosenRarity)
        return "{" .. itemName .. ", " .. color .. "}"
    end
    return text:gsub("{(.-)}", coloringLootName)
end

function ItemsDatabase.setTier(widget, item)
    if not g_game.getFeature(GameItemTier) or not widget then
        return
    end
    local tier = type(item) == "number" and item or (item and item:getTier()) or 0
    if tier and tier > 0 then
        tier = math.min(math.max(tonumber(tier) or 0, 1), 20)
        widget.tier:setText(tostring(tier))
        widget.tier:setVisible(true)
    else
        widget.tier:setText("")
        widget.tier:setVisible(false)
    end
end

function ItemsDatabase.setCharges(widget, item, style)
    if not g_game.getFeature(GameThingCounter) or not widget then
        return
    end

    if item and item:getCharges() > 0 then
        widget.charges:setText(item:getCharges())
    else
        widget.charges:setText("")
    end

    if style then
        widget:setStyle(style)
    end
end


function ItemsDatabase.setDuration(widget, item, style)
    if not g_game.getFeature(GameThingClock) or not widget then
        return
    end

    if item and item:getDurationTime() > 0 then
            local durationTimeLeft = item:getDurationTime()
            widget.duration:setText(string.format("%dm%02d", durationTimeLeft / 60, durationTimeLeft % 60))
    else
        widget.duration:setText("")
    end

    if style then
        widget:setStyle(style)
    end
end
